<?php
/**
 * Webhook queue class for Floyi Connect.
 *
 * Handles queuing and retrying failed webhook notifications to Floyi.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Webhook Queue class.
 */
class Floyi_Webhook_Queue {

    /**
     * Maximum retry attempts.
     */
    const MAX_RETRIES = 3;

    /**
     * Retry intervals in seconds (1min, 5min, 15min).
     */
    const RETRY_INTERVALS = array(60, 300, 900);

    /**
     * Add event to webhook queue.
     *
     * @param string $event_type Event type.
     * @param array  $payload    Event payload.
     * @return int|false Queue item ID or false on failure.
     */
    public static function add($event_type, $payload) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        $result = $wpdb->insert(
            $table,
            array(
                'event_type' => sanitize_key($event_type),
                'payload' => wp_json_encode($payload),
                'status' => 'pending',
                'next_retry' => current_time('mysql', true),
            ),
            array('%s', '%s', '%s', '%s')
        );

        if (!$result) {
            return false;
        }

        return $wpdb->insert_id;
    }

    /**
     * Process pending webhooks in the queue.
     */
    public static function process_queue() {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        error_log("[Floyi Queue] process_queue called, table={$table}");

        // Get pending items ready for retry
        $now = current_time('mysql', true);
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table
             WHERE status IN ('pending', 'retrying')
             AND (next_retry IS NULL OR next_retry <= %s)
             ORDER BY created_at ASC
             LIMIT 10",
            $now
        ));

        error_log("[Floyi Queue] Found " . count($items) . " items to process (now={$now})");

        foreach ($items as $item) {
            error_log("[Floyi Queue] Processing item id={$item->id}, event={$item->event_type}, status={$item->status}");
            self::process_item($item);
        }

        // Cleanup old completed/failed items (older than 7 days)
        self::cleanup_old_items();
    }

    /**
     * Process a single queue item.
     *
     * @param object $item Queue item.
     */
    private static function process_item($item) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        // Decode payload
        $payload = json_decode($item->payload, true);

        if (!$payload) {
            self::mark_failed($item->id, 'Invalid payload');
            return;
        }

        // Attempt to send webhook
        $result = self::send_webhook($item->event_type, $payload);

        if ($result === true) {
            // Success - mark as completed
            $wpdb->update(
                $table,
                array('status' => 'completed'),
                array('id' => $item->id),
                array('%s'),
                array('%d')
            );
        } else {
            // Failed - check if we should retry
            $attempts = intval($item->attempts) + 1;

            if ($attempts >= self::MAX_RETRIES) {
                self::mark_failed($item->id, $result);
            } else {
                // Schedule retry
                $retry_interval = self::RETRY_INTERVALS[$attempts - 1] ?? self::RETRY_INTERVALS[count(self::RETRY_INTERVALS) - 1];
                $next_retry = gmdate('Y-m-d H:i:s', time() + $retry_interval);

                $wpdb->update(
                    $table,
                    array(
                        'attempts' => $attempts,
                        'status' => 'retrying',
                        'next_retry' => $next_retry,
                        'last_error' => $result,
                    ),
                    array('id' => $item->id),
                    array('%d', '%s', '%s', '%s'),
                    array('%d')
                );
            }
        }
    }

    /**
     * Send webhook to Floyi.
     *
     * @param string $event_type Event type.
     * @param array  $payload    Event payload.
     * @return true|string True on success, error message on failure.
     */
    private static function send_webhook($event_type, $payload) {
        $floyi_url = Floyi_Settings::get_floyi_url();
        $webhook_url = trailingslashit($floyi_url) . 'api/cms/wordpress/webhook/receive/';

        $site_token = Floyi_Settings::get_site_token();
        $webhook_secret = Floyi_Settings::get_webhook_secret();

        error_log("[Floyi Queue] send_webhook: event={$event_type}, url={$webhook_url}");
        error_log("[Floyi Queue] send_webhook: site_token=" . (empty($site_token) ? "EMPTY" : "set(" . strlen($site_token) . " chars)"));
        error_log("[Floyi Queue] send_webhook: webhook_secret=" . (empty($webhook_secret) ? "EMPTY" : "set(" . strlen($webhook_secret) . " chars)"));

        if (!$site_token || !$webhook_secret) {
            error_log("[Floyi Queue] send_webhook: ABORT - Not connected (missing token or secret)");
            return 'Not connected to Floyi';
        }

        // Prepare payload
        $body = wp_json_encode(array(
            'event' => $event_type,
            'data' => $payload,
            'site_id' => Floyi_Settings::get_site_id(),
            'timestamp' => current_time('c'),
        ));

        error_log("[Floyi Queue] send_webhook: site_id=" . Floyi_Settings::get_site_id());
        error_log("[Floyi Queue] send_webhook: body_length=" . strlen($body));

        // Generate signature
        $timestamp = (string) time();
        $nonce = bin2hex(random_bytes(16));
        $signature = Floyi_Security::generate_signature($body, $timestamp, $nonce, $webhook_secret);

        // Send request
        $response = wp_remote_post($webhook_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Floyi-Token' => $site_token,
                'X-Floyi-Timestamp' => $timestamp,
                'X-Floyi-Nonce' => $nonce,
                'X-Floyi-Signature' => $signature,
            ),
            'body' => $body,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            error_log("[Floyi Queue] send_webhook: WP_Error - " . $response->get_error_message());
            return $response->get_error_message();
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        error_log("[Floyi Queue] send_webhook: Response status={$status_code}, body=" . substr($response_body, 0, 500));

        if ($status_code >= 200 && $status_code < 300) {
            return true;
        }

        return "HTTP $status_code: $response_body";
    }

    /**
     * Mark item as permanently failed.
     *
     * @param int    $id    Item ID.
     * @param string $error Error message.
     */
    private static function mark_failed($id, $error) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        $wpdb->update(
            $table,
            array(
                'status' => 'failed',
                'last_error' => $error,
            ),
            array('id' => $id),
            array('%s', '%s'),
            array('%d')
        );
    }

    /**
     * Cleanup old queue items.
     */
    private static function cleanup_old_items() {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';
        $cutoff = gmdate('Y-m-d H:i:s', strtotime('-7 days'));

        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table
             WHERE status IN ('completed', 'failed')
             AND updated_at < %s",
            $cutoff
        ));
    }

    /**
     * Get queue status for admin display.
     *
     * @return array Queue status.
     */
    public static function get_status() {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        $stats = $wpdb->get_results(
            "SELECT status, COUNT(*) as count FROM $table GROUP BY status"
        );

        $result = array(
            'pending' => 0,
            'retrying' => 0,
            'completed' => 0,
            'failed' => 0,
        );

        foreach ($stats as $stat) {
            $result[$stat->status] = intval($stat->count);
        }

        return $result;
    }

    /**
     * Get failed items for manual retry.
     *
     * @param int $limit Maximum items to return.
     * @return array Failed items.
     */
    public static function get_failed_items($limit = 20) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE status = 'failed' ORDER BY created_at DESC LIMIT %d",
            $limit
        ));
    }

    /**
     * Manually retry a failed item.
     *
     * @param int $id Item ID.
     * @return bool True on success.
     */
    public static function retry_item($id) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_webhook_queue';

        return $wpdb->update(
            $table,
            array(
                'attempts' => 0,
                'status' => 'pending',
                'next_retry' => current_time('mysql', true),
                'last_error' => null,
            ),
            array('id' => $id),
            array('%d', '%s', '%s', '%s'),
            array('%d')
        ) !== false;
    }
}
