<?php
/**
 * Settings class for Floyi Connect.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Settings class.
 */
class Floyi_Settings {

    /**
     * Option keys.
     */
    const OPTION_SITE_ID = 'floyi_site_id';
    const OPTION_SITE_TOKEN = 'floyi_site_token';
    const OPTION_WEBHOOK_SECRET = 'floyi_webhook_secret';
    const OPTION_CONNECTION_STATUS = 'floyi_connection_status';
    const OPTION_CONNECTED_AT = 'floyi_connected_at';
    const OPTION_FLOYI_URL = 'floyi_app_url';
    const OPTION_LAST_SYNC = 'floyi_last_sync';
    const OPTION_CONNECTION_CODE = 'floyi_connection_code';
    const OPTION_CONNECTION_CODE_EXPIRY = 'floyi_connection_code_expiry';

    /**
     * Default production URL.
     */
    const DEFAULT_URL = 'https://app.floyi.com';

    /**
     * Get site ID.
     *
     * @return string Site ID.
     */
    public static function get_site_id() {
        $site_id = get_option(self::OPTION_SITE_ID);

        if (!$site_id) {
            $site_id = wp_generate_uuid4();
            update_option(self::OPTION_SITE_ID, $site_id);
        }

        return $site_id;
    }

    /**
     * Get site token.
     *
     * @return string|false Site token or false if not set.
     */
    public static function get_site_token() {
        return get_option(self::OPTION_SITE_TOKEN, false);
    }

    /**
     * Set site token.
     *
     * @param string $token The site token.
     */
    public static function set_site_token($token) {
        update_option(self::OPTION_SITE_TOKEN, $token);
    }

    /**
     * Get webhook secret.
     *
     * @return string|false Webhook secret or false if not set.
     */
    public static function get_webhook_secret() {
        return get_option(self::OPTION_WEBHOOK_SECRET, false);
    }

    /**
     * Set webhook secret.
     *
     * @param string $secret The webhook secret.
     */
    public static function set_webhook_secret($secret) {
        update_option(self::OPTION_WEBHOOK_SECRET, $secret);
    }

    /**
     * Get connection status.
     *
     * @return string Connection status ('connected', 'disconnected', 'pending').
     */
    public static function get_connection_status() {
        return get_option(self::OPTION_CONNECTION_STATUS, 'disconnected');
    }

    /**
     * Set connection status.
     *
     * @param string $status The connection status.
     */
    public static function set_connection_status($status) {
        update_option(self::OPTION_CONNECTION_STATUS, $status);
    }

    /**
     * Check if connected.
     *
     * @return bool True if connected.
     */
    public static function is_connected() {
        return self::get_connection_status() === 'connected';
    }

    /**
     * Get Floyi app URL.
     *
     * Priority:
     * 1. FLOYI_API_URL constant in wp-config.php (for staging/dev overrides)
     * 2. Stored URL from last connection
     * 3. Production default
     *
     * To use a staging or development server, add this to wp-config.php:
     *   define('FLOYI_API_URL', 'https://staging-dev.floyi.com');
     *
     * @return string Floyi app URL (no trailing slash).
     */
    public static function get_floyi_url() {
        // wp-config.php constant takes highest priority (staging/dev override)
        if (defined('FLOYI_API_URL') && FLOYI_API_URL) {
            return rtrim(FLOYI_API_URL, '/');
        }

        // Fall back to stored URL (set during connection)
        $stored_url = get_option(self::OPTION_FLOYI_URL, '');
        if ($stored_url) {
            return rtrim($stored_url, '/');
        }

        return self::DEFAULT_URL;
    }

    /**
     * Set Floyi app URL.
     *
     * @param string $url The Floyi app URL.
     */
    public static function set_floyi_url($url) {
        update_option(self::OPTION_FLOYI_URL, rtrim($url, '/'));
    }

    /**
     * Generate and store a new connection code.
     *
     * @param int $expiry_minutes Code expiry in minutes (default 10).
     * @return string The connection code.
     */
    public static function generate_connection_code($expiry_minutes = 10) {
        $code = Floyi_Security::generate_connection_code();
        $expiry = time() + ($expiry_minutes * 60);

        update_option(self::OPTION_CONNECTION_CODE, $code);
        update_option(self::OPTION_CONNECTION_CODE_EXPIRY, $expiry);

        return $code;
    }

    /**
     * Get current connection code.
     *
     * @return string|false Connection code or false if expired/not set.
     */
    public static function get_connection_code() {
        $code = get_option(self::OPTION_CONNECTION_CODE);
        $expiry = get_option(self::OPTION_CONNECTION_CODE_EXPIRY);

        if (!$code || !$expiry || time() > $expiry) {
            return false;
        }

        return $code;
    }

    /**
     * Validate connection code.
     *
     * @param string $code The code to validate.
     * @return bool True if valid.
     */
    public static function validate_connection_code($code) {
        $stored_code = self::get_connection_code();

        if (!$stored_code) {
            return false;
        }

        return hash_equals($stored_code, strtoupper($code));
    }

    /**
     * Clear connection code after use.
     */
    public static function clear_connection_code() {
        delete_option(self::OPTION_CONNECTION_CODE);
        delete_option(self::OPTION_CONNECTION_CODE_EXPIRY);
    }

    /**
     * Mark as connected.
     *
     * @param string $site_id   Site ID for this WordPress site.
     * @param string $token     Site token for authenticating Floyi requests.
     * @param string $secret    Webhook secret for signing webhooks.
     * @param string $floyi_url Floyi app URL used for the connection.
     */
    public static function mark_connected($site_id, $token, $secret, $floyi_url = null) {
        update_option(self::OPTION_SITE_ID, $site_id);
        self::set_site_token($token);
        self::set_webhook_secret($secret);
        self::set_connection_status('connected');
        update_option(self::OPTION_CONNECTED_AT, current_time('mysql'));

        if ($floyi_url) {
            self::set_floyi_url($floyi_url);
        }

        self::clear_connection_code();
    }

    /**
     * Mark as disconnected.
     */
    public static function mark_disconnected() {
        self::set_connection_status('disconnected');
        delete_option(self::OPTION_SITE_TOKEN);
        delete_option(self::OPTION_WEBHOOK_SECRET);
        delete_option(self::OPTION_CONNECTED_AT);
    }

    /**
     * Update last sync time.
     */
    public static function update_last_sync() {
        update_option(self::OPTION_LAST_SYNC, current_time('mysql'));
    }

    /**
     * Get last sync time.
     *
     * @return string|false Last sync time or false if never synced.
     */
    public static function get_last_sync() {
        return get_option(self::OPTION_LAST_SYNC, false);
    }

    /**
     * Get all settings for display.
     *
     * @return array Settings array.
     */
    public static function get_all_settings() {
        return array(
            'site_id' => self::get_site_id(),
            'connection_status' => self::get_connection_status(),
            'is_connected' => self::is_connected(),
            'connected_at' => get_option(self::OPTION_CONNECTED_AT, ''),
            'last_sync' => self::get_last_sync(),
            'floyi_url' => self::get_floyi_url(),
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
        );
    }
}
