<?php
/**
 * Security class for Floyi Connect.
 *
 * Handles HMAC signature verification and nonce cache for replay protection.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Security class.
 */
class Floyi_Security {

    /**
     * Maximum age for signatures (5 minutes).
     */
    const MAX_TIMESTAMP_AGE = 300;

    /**
     * Nonce expiry time (10 minutes).
     */
    const NONCE_EXPIRY = 600;

    /**
     * Verify HMAC signature from Floyi request.
     *
     * @param WP_REST_Request $request The REST request object.
     * @return bool|WP_Error True if valid, WP_Error if invalid.
     */
    public static function verify_signature($request) {
        $token = $request->get_header('X-Floyi-Token');
        $timestamp = $request->get_header('X-Floyi-Timestamp');
        $nonce = $request->get_header('X-Floyi-Nonce');
        $signature = $request->get_header('X-Floyi-Signature');

        // Check all required headers are present
        if (!$token || !$timestamp || !$nonce || !$signature) {
            return new WP_Error(
                'missing_auth_headers',
                __('Missing authentication headers.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Verify token matches stored token
        $stored_token = get_option('floyi_site_token');
        if (!$stored_token || !hash_equals($stored_token, $token)) {
            return new WP_Error(
                'invalid_token',
                __('Invalid authentication token.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Verify timestamp is recent (prevent replay attacks)
        $current_time = time();
        $request_time = intval($timestamp);
        if (abs($current_time - $request_time) > self::MAX_TIMESTAMP_AGE) {
            return new WP_Error(
                'expired_timestamp',
                __('Request timestamp has expired.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Check nonce hasn't been used before
        if (self::nonce_exists($nonce)) {
            return new WP_Error(
                'nonce_reused',
                __('Request nonce has already been used.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Verify HMAC signature
        $webhook_secret = get_option('floyi_webhook_secret');
        if (!$webhook_secret) {
            return new WP_Error(
                'no_webhook_secret',
                __('Webhook secret not configured.', 'floyi-connect'),
                array('status' => 500)
            );
        }

        $body = $request->get_body();
        $expected_signature = self::generate_signature($body, $timestamp, $nonce, $webhook_secret);

        if (!hash_equals($expected_signature, $signature)) {
            return new WP_Error(
                'invalid_signature',
                __('Invalid request signature.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Store nonce to prevent replay
        self::store_nonce($nonce);

        return true;
    }

    /**
     * Generate HMAC signature.
     *
     * @param string $payload Request payload.
     * @param string $timestamp Unix timestamp.
     * @param string $nonce Request nonce.
     * @param string $secret Webhook secret.
     * @return string HMAC signature.
     */
    public static function generate_signature($payload, $timestamp, $nonce, $secret) {
        $message = $timestamp . '.' . $nonce . '.' . $payload;
        return hash_hmac('sha256', $message, $secret);
    }

    /**
     * Check if nonce exists in cache.
     *
     * @param string $nonce The nonce to check.
     * @return bool True if nonce exists.
     */
    public static function nonce_exists($nonce) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_nonce_cache';
        $nonce_hash = hash('sha256', $nonce);

        $result = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE nonce = %s",
            $nonce_hash
        ));

        return intval($result) > 0;
    }

    /**
     * Store nonce in cache.
     *
     * @param string $nonce The nonce to store.
     */
    public static function store_nonce($nonce) {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_nonce_cache';
        $nonce_hash = hash('sha256', $nonce);

        $wpdb->insert(
            $table,
            array(
                'nonce' => $nonce_hash,
                'created_at' => current_time('mysql', true),
            ),
            array('%s', '%s')
        );

        // Cleanup old nonces
        self::cleanup_expired_nonces();
    }

    /**
     * Clean up expired nonces.
     */
    public static function cleanup_expired_nonces() {
        global $wpdb;

        $table = $wpdb->prefix . 'floyi_nonce_cache';
        $expiry_time = gmdate('Y-m-d H:i:s', time() - self::NONCE_EXPIRY);

        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE created_at < %s",
            $expiry_time
        ));
    }

    /**
     * Generate a secure random token.
     *
     * @param int $length Token length in bytes.
     * @return string Hex-encoded token.
     */
    public static function generate_token($length = 32) {
        return bin2hex(random_bytes($length));
    }

    /**
     * Generate connection code for handshake (XXXX-XXXX-XXXX format).
     *
     * @return string Connection code.
     */
    public static function generate_connection_code() {
        $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $code_parts = array();

        for ($i = 0; $i < 3; $i++) {
            $part = '';
            for ($j = 0; $j < 4; $j++) {
                $part .= $chars[random_int(0, strlen($chars) - 1)];
            }
            $code_parts[] = $part;
        }

        return implode('-', $code_parts);
    }

    /**
     * Sanitize and validate URL.
     *
     * @param string $url The URL to validate.
     * @return string|false Sanitized URL or false if invalid.
     */
    public static function validate_url($url) {
        $url = esc_url_raw($url);

        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return false;
        }

        // Ensure HTTPS
        if (strpos($url, 'https://') !== 0) {
            return false;
        }

        return $url;
    }
}
