<?php
/**
 * Publisher class for Floyi Connect.
 *
 * Handles creating, updating, and managing WordPress posts from Floyi.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Publisher class.
 */
class Floyi_Publisher {

    /**
     * Create a new post.
     *
     * @param array $params Post parameters.
     * @return array|WP_Error Post data or error.
     */
    public static function create_post($params) {
        // Validate required fields
        if (empty($params['title'])) {
            return new WP_Error(
                'missing_title',
                __('Post title is required.', 'floyi-connect'),
                array('status' => 400)
            );
        }

        // Prepare post data
        $post_data = array(
            'post_title' => sanitize_text_field($params['title']),
            'post_content' => isset($params['content']) ? wp_kses_post($params['content']) : '',
            'post_status' => isset($params['status']) ? sanitize_key($params['status']) : 'draft',
            'post_type' => isset($params['type']) ? sanitize_key($params['type']) : 'post',
        );

        // Optional fields
        if (!empty($params['slug'])) {
            $post_data['post_name'] = sanitize_title($params['slug']);
        }

        if (!empty($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }

        if (!empty($params['author'])) {
            $post_data['post_author'] = intval($params['author']);
        }

        // Parent page support for hierarchical post types (pages)
        if (!empty($params['parent'])) {
            $post_data['post_parent'] = intval($params['parent']);
        }

        if (!empty($params['date'])) {
            $post_data['post_date'] = sanitize_text_field($params['date']);
            $post_data['post_date_gmt'] = get_gmt_from_date($params['date']);
        }

        // Insert post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return $post_id;
        }

        // Set categories
        if (!empty($params['categories']) && is_array($params['categories'])) {
            wp_set_post_categories($post_id, array_map('intval', $params['categories']));
        }

        // Set tags
        if (!empty($params['tags']) && is_array($params['tags'])) {
            wp_set_post_tags($post_id, array_map('intval', $params['tags']));
        }

        // Set featured image
        if (!empty($params['featured_media'])) {
            set_post_thumbnail($post_id, intval($params['featured_media']));
        }

        // Set SEO fields
        if (!empty($params['seo'])) {
            self::set_seo_fields($post_id, $params['seo']);
        }

        // Set custom meta fields
        if (!empty($params['meta']) && is_array($params['meta'])) {
            foreach ($params['meta'] as $key => $value) {
                update_post_meta($post_id, sanitize_key($key), $value);
            }
        }

        // Store Floyi reference
        if (!empty($params['floyi_article_id'])) {
            update_post_meta($post_id, '_floyi_article_id', sanitize_text_field($params['floyi_article_id']));
        }

        // Get and return post data
        $post = get_post($post_id);
        return self::format_post_response($post);
    }

    /**
     * Update an existing post.
     *
     * @param int   $post_id Post ID.
     * @param array $params  Update parameters.
     * @return array|WP_Error Post data or error.
     */
    public static function update_post($post_id, $params) {
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found.', 'floyi-connect'),
                array('status' => 404)
            );
        }

        // Prepare update data
        $post_data = array('ID' => $post_id);

        if (isset($params['title'])) {
            $post_data['post_title'] = sanitize_text_field($params['title']);
        }

        if (isset($params['content'])) {
            $post_data['post_content'] = wp_kses_post($params['content']);
        }

        if (isset($params['status'])) {
            $post_data['post_status'] = sanitize_key($params['status']);
        }

        if (isset($params['slug'])) {
            $post_data['post_name'] = sanitize_title($params['slug']);
        }

        if (isset($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }

        if (isset($params['author'])) {
            $post_data['post_author'] = intval($params['author']);
        }

        // Parent page support for hierarchical post types (pages)
        if (isset($params['parent'])) {
            $post_data['post_parent'] = intval($params['parent']);
        }

        // Update post
        $result = wp_update_post($post_data, true);

        if (is_wp_error($result)) {
            return $result;
        }

        // Update categories
        if (isset($params['categories']) && is_array($params['categories'])) {
            wp_set_post_categories($post_id, array_map('intval', $params['categories']));
        }

        // Update tags
        if (isset($params['tags']) && is_array($params['tags'])) {
            wp_set_post_tags($post_id, array_map('intval', $params['tags']));
        }

        // Update featured image
        if (isset($params['featured_media'])) {
            if ($params['featured_media']) {
                set_post_thumbnail($post_id, intval($params['featured_media']));
            } else {
                delete_post_thumbnail($post_id);
            }
        }

        // Update SEO fields
        if (!empty($params['seo'])) {
            self::set_seo_fields($post_id, $params['seo']);
        }

        // Update custom meta fields
        if (!empty($params['meta']) && is_array($params['meta'])) {
            foreach ($params['meta'] as $key => $value) {
                update_post_meta($post_id, sanitize_key($key), $value);
            }
        }

        // Get and return updated post data
        $post = get_post($post_id);
        return self::format_post_response($post);
    }

    /**
     * Set SEO fields based on detected plugin.
     *
     * @param int   $post_id Post ID.
     * @param array $seo     SEO fields.
     */
    public static function set_seo_fields($post_id, $seo) {
        $seo_plugin = Floyi_Capabilities::get_seo_plugin();

        if (!$seo_plugin) {
            return;
        }

        $title = isset($seo['title']) ? sanitize_text_field($seo['title']) : '';
        $description = isset($seo['description']) ? sanitize_textarea_field($seo['description']) : '';
        $focus_keyword = isset($seo['focus_keyword']) ? sanitize_text_field($seo['focus_keyword']) : '';

        switch ($seo_plugin['slug']) {
            case 'yoast':
                if ($title) update_post_meta($post_id, '_yoast_wpseo_title', $title);
                if ($description) update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
                if ($focus_keyword) update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword);
                break;

            case 'rankmath':
                if ($title) update_post_meta($post_id, 'rank_math_title', $title);
                if ($description) update_post_meta($post_id, 'rank_math_description', $description);
                if ($focus_keyword) update_post_meta($post_id, 'rank_math_focus_keyword', $focus_keyword);
                break;

            case 'aioseo':
                // AIOSEO stores data in its own custom table (wp_aioseo_posts), not in postmeta.
                // Direct table access via $wpdb is the most reliable approach across AIOSEO versions.
                global $wpdb;
                $table = $wpdb->prefix . 'aioseo_posts';
                $existing = $wpdb->get_row($wpdb->prepare("SELECT id, keyphrases FROM {$table} WHERE post_id = %d", $post_id));

                $data = array();
                if ($title) $data['title'] = $title;
                if ($description) $data['description'] = $description;
                if ($focus_keyword) {
                    // Merge with existing keyphrases if present
                    $keyphrases = array();
                    if ($existing && !empty($existing->keyphrases)) {
                        $decoded = json_decode($existing->keyphrases, true);
                        if (is_array($decoded)) $keyphrases = $decoded;
                    }
                    $keyphrases['focus'] = array('keyphrase' => $focus_keyword, 'score' => 0, 'analysis' => new \stdClass());
                    $data['keyphrases'] = wp_json_encode($keyphrases);
                }

                if ($existing) {
                    $data['updated'] = current_time('mysql');
                    $wpdb->update($table, $data, array('post_id' => $post_id));
                } else {
                    $data['post_id'] = $post_id;
                    $data['created'] = current_time('mysql');
                    $data['updated'] = current_time('mysql');
                    $wpdb->insert($table, $data);
                }
                break;

            case 'seopress':
                if ($title) update_post_meta($post_id, '_seopress_titles_title', $title);
                if ($description) update_post_meta($post_id, '_seopress_titles_desc', $description);
                if ($focus_keyword) update_post_meta($post_id, '_seopress_analysis_target_kw', $focus_keyword);
                break;
        }
    }

    /**
     * Sideload media from URL.
     *
     * @param string $url      Media URL.
     * @param string $filename Optional filename.
     * @param string $alt      Alt text.
     * @param string $caption  Caption.
     * @return array|WP_Error Media data or error.
     */
    public static function sideload_media($url, $filename = null, $alt = '', $caption = '') {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Download file to temp location
        $temp_file = download_url($url, 30);

        if (is_wp_error($temp_file)) {
            return new WP_Error(
                'download_failed',
                __('Failed to download media file.', 'floyi-connect'),
                array('status' => 400)
            );
        }

        // Get filename from URL if not provided
        if (!$filename) {
            $filename = basename(wp_parse_url($url, PHP_URL_PATH));
        }

        // Prepare file array
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file,
        );

        // Sideload into media library
        $attachment_id = media_handle_sideload($file_array, 0);

        // Delete temp file if still exists
        if (file_exists($temp_file)) {
            @unlink($temp_file);
        }

        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        // Set alt text
        if ($alt) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
        }

        // Set caption
        if ($caption) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_excerpt' => $caption,
            ));
        }

        // Get attachment data
        $attachment = get_post($attachment_id);
        $attachment_url = wp_get_attachment_url($attachment_id);

        return array(
            'id' => $attachment_id,
            'source_url' => $attachment_url,
            'title' => $attachment->post_title,
            'alt_text' => $alt,
            'caption' => $caption,
            'mime_type' => $attachment->post_mime_type,
        );
    }

    /**
     * Format post data for API response.
     *
     * @param WP_Post $post Post object.
     * @return array Formatted post data.
     */
    public static function format_post_response($post) {
        $featured_image_id = get_post_thumbnail_id($post->ID);
        $featured_image_url = $featured_image_id ? wp_get_attachment_url($featured_image_id) : null;

        return array(
            'id' => $post->ID,
            'title' => $post->post_title,
            'content' => $post->post_content,
            'excerpt' => $post->post_excerpt,
            'status' => $post->post_status,
            'type' => $post->post_type,
            'slug' => $post->post_name,
            'link' => get_permalink($post->ID),
            'edit_link' => get_edit_post_link($post->ID, 'raw'),
            'preview_link' => get_preview_post_link($post->ID),
            'author' => intval($post->post_author),
            'featured_media' => $featured_image_id ? intval($featured_image_id) : null,
            'featured_media_url' => $featured_image_url,
            'categories' => wp_get_post_categories($post->ID),
            'tags' => wp_get_post_tags($post->ID, array('fields' => 'ids')),
            'date' => $post->post_date,
            'modified' => $post->post_modified,
            'floyi_article_id' => get_post_meta($post->ID, '_floyi_article_id', true),
        );
    }
}
