<?php
/**
 * Capabilities detection class for Floyi Connect.
 *
 * Detects and reports WordPress site capabilities including post types,
 * taxonomies, SEO plugins, and other features.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Capabilities class.
 */
class Floyi_Capabilities {

    /**
     * Known SEO plugins.
     */
    const SEO_PLUGINS = array(
        'wordpress-seo/wp-seo.php' => array(
            'slug' => 'yoast',
            'name' => 'Yoast SEO',
        ),
        'seo-by-rank-math/rank-math.php' => array(
            'slug' => 'rankmath',
            'name' => 'Rank Math',
        ),
        'all-in-one-seo-pack/all_in_one_seo_pack.php' => array(
            'slug' => 'aioseo',
            'name' => 'All in One SEO',
        ),
        'wp-seopress/seopress.php' => array(
            'slug' => 'seopress',
            'name' => 'SEOPress',
        ),
    );

    /**
     * Get all site capabilities.
     *
     * @return array Capabilities array.
     */
    public static function get_all() {
        return array(
            'post_types' => self::get_post_types(),
            'categories' => self::get_categories(),
            'tags' => self::get_tags(),
            'authors' => self::get_authors(),
            'pages' => self::get_pages(),
            'seo_plugin' => self::get_seo_plugin(),
            'editor_type' => self::get_editor_type(),
            'custom_fields' => self::get_custom_fields(),
            'media' => self::get_media_capabilities(),
            'wordpress' => self::get_wordpress_info(),
        );
    }

    /**
     * Get available post types.
     *
     * @return array Post types array.
     */
    public static function get_post_types() {
        $post_types = array();
        $types = get_post_types(array('show_in_rest' => true), 'objects');

        foreach ($types as $type) {
            // Skip internal/system post types that shouldn't appear in publishing UI
            if (in_array($type->name, array(
                'attachment', 'revision', 'nav_menu_item', 'custom_css',
                'customize_changeset', 'oembed_cache', 'user_request',
                'wp_block', 'wp_template', 'wp_template_part',
                'wp_global_styles', 'wp_navigation',
                'wp_font_family', 'wp_font_face',
                'rm_content_editor', 'schema', 'rank_math_schema',
            ), true)) {
                continue;
            }

            $post_types[] = array(
                'slug' => $type->name,
                'name' => $type->label,
                'singular' => $type->labels->singular_name,
                'supports' => get_all_post_type_supports($type->name),
                'hierarchical' => $type->hierarchical,
                'has_archive' => $type->has_archive,
            );
        }

        return $post_types;
    }

    /**
     * Get available categories.
     *
     * @return array Categories array.
     */
    public static function get_categories() {
        $categories = array();
        $terms = get_terms(array(
            'taxonomy' => 'category',
            'hide_empty' => false,
            'number' => 500,
        ));

        if (is_wp_error($terms)) {
            return $categories;
        }

        foreach ($terms as $term) {
            $categories[] = array(
                'id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug,
                'parent' => $term->parent,
                'count' => $term->count,
            );
        }

        return $categories;
    }

    /**
     * Get available tags.
     *
     * @return array Tags array.
     */
    public static function get_tags() {
        $tags = array();
        $terms = get_terms(array(
            'taxonomy' => 'post_tag',
            'hide_empty' => false,
            'number' => 500,
        ));

        if (is_wp_error($terms)) {
            return $tags;
        }

        foreach ($terms as $term) {
            $tags[] = array(
                'id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug,
                'count' => $term->count,
            );
        }

        return $tags;
    }

    /**
     * Get available authors.
     *
     * @return array Authors array.
     */
    public static function get_authors() {
        $authors = array();
        $users = get_users(array(
            'capability__in' => array('edit_posts'),
            'number' => 100,
        ));

        foreach ($users as $user) {
            $authors[] = array(
                'id' => $user->ID,
                'name' => $user->display_name,
                'email' => $user->user_email,
                'slug' => $user->user_nicename,
            );
        }

        return $authors;
    }

    /**
     * Get available pages for parent page selection.
     *
     * Returns published pages that can be used as parent pages
     * when creating new pages.
     *
     * @return array Pages array with id, title, and parent.
     */
    public static function get_pages() {
        $pages_list = array();
        $pages = get_pages(array(
            'post_status' => 'publish',
            'number' => 500,
            'sort_column' => 'menu_order,post_title',
            'sort_order' => 'ASC',
        ));

        if (empty($pages)) {
            return $pages_list;
        }

        foreach ($pages as $page) {
            $pages_list[] = array(
                'id' => $page->ID,
                'title' => $page->post_title,
                'slug' => $page->post_name,
                'parent' => $page->post_parent,
            );
        }

        return $pages_list;
    }

    /**
     * Detect installed SEO plugin.
     *
     * @return array|null SEO plugin info or null if not found.
     */
    public static function get_seo_plugin() {
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        foreach (self::SEO_PLUGINS as $plugin_file => $plugin_info) {
            if (is_plugin_active($plugin_file)) {
                $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_file);
                return array(
                    'slug' => $plugin_info['slug'],
                    'name' => $plugin_info['name'],
                    'version' => $plugin_data['Version'],
                );
            }
        }

        return null;
    }

    /**
     * Get editor type (Classic or Block/Gutenberg).
     *
     * @return string Editor type.
     */
    public static function get_editor_type() {
        // Check if Classic Editor plugin is active
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        if (is_plugin_active('classic-editor/classic-editor.php')) {
            $classic_mode = get_option('classic-editor-replace');
            if ($classic_mode === 'classic') {
                return 'classic';
            }
        }

        // Check WordPress version (Gutenberg was introduced in 5.0)
        global $wp_version;
        if (version_compare($wp_version, '5.0', '>=')) {
            return 'block';
        }

        return 'classic';
    }

    /**
     * Get available custom fields.
     *
     * @return array Custom fields array.
     */
    public static function get_custom_fields() {
        global $wpdb;

        $meta_keys = $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT meta_key FROM {$wpdb->postmeta}
             WHERE meta_key NOT LIKE %s
             AND meta_key NOT LIKE %s
             ORDER BY meta_key
             LIMIT 100",
            '\_%',
            'field\_%'
        ));

        return $meta_keys ?: array();
    }

    /**
     * Get media capabilities.
     *
     * @return array Media capabilities.
     */
    public static function get_media_capabilities() {
        return array(
            'max_upload_size' => wp_max_upload_size(),
            'allowed_mime_types' => get_allowed_mime_types(),
            'image_sizes' => self::get_image_sizes(),
        );
    }

    /**
     * Get registered image sizes.
     *
     * @return array Image sizes array.
     */
    public static function get_image_sizes() {
        $sizes = array();
        $registered_sizes = wp_get_registered_image_subsizes();

        foreach ($registered_sizes as $name => $size) {
            $sizes[] = array(
                'name' => $name,
                'width' => $size['width'],
                'height' => $size['height'],
                'crop' => $size['crop'],
            );
        }

        return $sizes;
    }

    /**
     * Get WordPress info.
     *
     * @return array WordPress info.
     */
    public static function get_wordpress_info() {
        global $wp_version;

        return array(
            'version' => $wp_version,
            'multisite' => is_multisite(),
            'site_url' => get_site_url(),
            'home_url' => get_home_url(),
            'admin_url' => admin_url(),
            'timezone' => wp_timezone_string(),
            'locale' => get_locale(),
        );
    }
}
