<?php
/**
 * REST API class for Floyi Connect.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * API class.
 */
class Floyi_API {

    /**
     * API namespace.
     */
    const NAMESPACE = 'floyi/v1';

    /**
     * Register REST API routes.
     */
    public static function register_routes() {
        // Health check (public)
        register_rest_route(self::NAMESPACE, '/health', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'health_check'),
            'permission_callback' => '__return_true',
        ));

        // Handshake - initiate connection (requires connection code)
        register_rest_route(self::NAMESPACE, '/handshake', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'handshake'),
            'permission_callback' => '__return_true',
        ));

        // Capabilities (requires authentication)
        register_rest_route(self::NAMESPACE, '/capabilities', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_capabilities'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));

        // Create post
        register_rest_route(self::NAMESPACE, '/posts', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'create_post'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));

        // Update post
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array(__CLASS__, 'update_post'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Get post
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_post'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Delete post
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array(__CLASS__, 'delete_post'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // List posts
        register_rest_route(self::NAMESPACE, '/posts', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'list_posts'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));

        // Media sideload
        register_rest_route(self::NAMESPACE, '/media/sideload', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'sideload_media'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));

        // Webhook receive (from Floyi)
        register_rest_route(self::NAMESPACE, '/webhook/receive', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'receive_webhook'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));

        // Disconnect
        register_rest_route(self::NAMESPACE, '/disconnect', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'disconnect'),
            'permission_callback' => array(__CLASS__, 'verify_request'),
        ));
    }

    /**
     * Verify request signature.
     *
     * @param WP_REST_Request $request The request object.
     * @return bool|WP_Error True if valid, WP_Error if invalid.
     */
    public static function verify_request($request) {
        $result = Floyi_Security::verify_signature($request);

        if (is_wp_error($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Health check endpoint.
     *
     * @return WP_REST_Response Response object.
     */
    public static function health_check() {
        return new WP_REST_Response(array(
            'status' => 'ok',
            'version' => FLOYI_CONNECT_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => phpversion(),
            'connected' => Floyi_Settings::is_connected(),
            'timestamp' => current_time('c'),
        ), 200);
    }

    /**
     * Handshake endpoint - establish connection with Floyi.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function handshake($request) {
        $params = $request->get_json_params();

        // Validate required fields
        $required = array('token', 'site_token', 'webhook_secret');
        foreach ($required as $field) {
            if (empty($params[$field])) {
                return new WP_Error(
                    'missing_field',
                    sprintf(__('Missing required field: %s', 'floyi-connect'), $field),
                    array('status' => 400)
                );
            }
        }

        // Validate connection code/token
        if (!Floyi_Settings::validate_connection_code($params['token'])) {
            return new WP_Error(
                'invalid_token',
                __('Invalid or expired connection code.', 'floyi-connect'),
                array('status' => 401)
            );
        }

        // Store credentials and mark as connected
        Floyi_Settings::mark_connected(
            Floyi_Settings::get_site_id(),
            sanitize_text_field($params['site_token']),
            sanitize_text_field($params['webhook_secret']),
            isset($params['floyi_url']) ? esc_url_raw($params['floyi_url']) : null
        );

        // Get capabilities to send back
        $capabilities = Floyi_Capabilities::get_all();

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Connection established successfully.', 'floyi-connect'),
            'site_id' => Floyi_Settings::get_site_id(),
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'capabilities' => $capabilities,
        ), 200);
    }

    /**
     * Get site capabilities.
     *
     * @return WP_REST_Response Response object.
     */
    public static function get_capabilities() {
        $capabilities = Floyi_Capabilities::get_all();

        return new WP_REST_Response($capabilities, 200);
    }

    /**
     * Create a new post.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function create_post($request) {
        $params = $request->get_json_params();

        $result = Floyi_Publisher::create_post($params);

        if (is_wp_error($result)) {
            return $result;
        }

        Floyi_Settings::update_last_sync();

        return new WP_REST_Response($result, 201);
    }

    /**
     * Update an existing post.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function update_post($request) {
        $post_id = intval($request->get_param('id'));
        $params = $request->get_json_params();

        $result = Floyi_Publisher::update_post($post_id, $params);

        if (is_wp_error($result)) {
            return $result;
        }

        Floyi_Settings::update_last_sync();

        return new WP_REST_Response($result, 200);
    }

    /**
     * Get a single post.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function get_post($request) {
        $post_id = intval($request->get_param('id'));
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found.', 'floyi-connect'),
                array('status' => 404)
            );
        }

        return new WP_REST_Response(Floyi_Publisher::format_post_response($post), 200);
    }

    /**
     * Delete a post.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function delete_post($request) {
        $post_id = intval($request->get_param('id'));
        $params = $request->get_json_params();
        $force = isset($params['force']) && $params['force'];

        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found.', 'floyi-connect'),
                array('status' => 404)
            );
        }

        $result = wp_delete_post($post_id, $force);

        if (!$result) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete post.', 'floyi-connect'),
                array('status' => 500)
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'message' => $force ? __('Post permanently deleted.', 'floyi-connect') : __('Post moved to trash.', 'floyi-connect'),
        ), 200);
    }

    /**
     * List posts.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response Response object.
     */
    public static function list_posts($request) {
        $args = array(
            'post_type' => $request->get_param('type') ?: 'post',
            'posts_per_page' => min(intval($request->get_param('per_page') ?: 100), 100),
            'paged' => max(intval($request->get_param('page') ?: 1), 1),
            'post_status' => $request->get_param('status') ?: 'any',
        );

        if ($search = $request->get_param('search')) {
            $args['s'] = sanitize_text_field($search);
        }

        $query = new WP_Query($args);
        $posts = array();

        foreach ($query->posts as $post) {
            $posts[] = Floyi_Publisher::format_post_response($post);
        }

        return new WP_REST_Response(array(
            'posts' => $posts,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
        ), 200);
    }

    /**
     * Sideload media from URL.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error Response object.
     */
    public static function sideload_media($request) {
        $params = $request->get_json_params();

        if (empty($params['source_url'])) {
            return new WP_Error(
                'missing_url',
                __('Source URL is required.', 'floyi-connect'),
                array('status' => 400)
            );
        }

        $result = Floyi_Publisher::sideload_media(
            esc_url_raw($params['source_url']),
            isset($params['filename']) ? sanitize_file_name($params['filename']) : null,
            isset($params['alt_text']) ? sanitize_text_field($params['alt_text']) : '',
            isset($params['caption']) ? sanitize_text_field($params['caption']) : ''
        );

        if (is_wp_error($result)) {
            return $result;
        }

        return new WP_REST_Response($result, 201);
    }

    /**
     * Receive webhook from Floyi.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response Response object.
     */
    public static function receive_webhook($request) {
        $params = $request->get_json_params();

        // Log webhook event (for debugging)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('Floyi webhook received: ' . wp_json_encode($params));
        }

        // Handle different event types
        $event = isset($params['event']) ? $params['event'] : '';

        switch ($event) {
            case 'content_updated':
                // Handle content update notification
                break;

            case 'connection_verified':
                // Handle connection verification
                break;

            default:
                // Unknown event type
                break;
        }

        return new WP_REST_Response(array(
            'success' => true,
            'received' => true,
        ), 200);
    }

    /**
     * Disconnect from Floyi.
     *
     * @return WP_REST_Response Response object.
     */
    public static function disconnect() {
        Floyi_Settings::mark_disconnected();

        return new WP_REST_Response(array(
            'success' => true,
            'message' => __('Disconnected from Floyi.', 'floyi-connect'),
        ), 200);
    }
}
