<?php
/**
 * Plugin Name: Floyi Connect
 * Plugin URI: https://floyi.com/wordpress-plugin
 * Description: Connect your WordPress site to Floyi for seamless content publishing from your topical authority platform.
 * Version: 1.0.0
 * Author: Floyi
 * Author URI: https://floyi.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: floyi-connect
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 *
 * @package Floyi_Connect
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('FLOYI_CONNECT_VERSION', '1.0.0');
define('FLOYI_CONNECT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('FLOYI_CONNECT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('FLOYI_CONNECT_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class.
 */
final class Floyi_Connect {

    /**
     * Plugin instance.
     *
     * @var Floyi_Connect
     */
    private static $instance = null;

    /**
     * Get plugin instance.
     *
     * @return Floyi_Connect
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required files.
     */
    private function load_dependencies() {
        // Core classes
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-settings.php';
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-api.php';
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-security.php';
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-capabilities.php';
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-publisher.php';
        require_once FLOYI_CONNECT_PLUGIN_DIR . 'includes/class-floyi-webhook-queue.php';

        // Admin classes
        if (is_admin()) {
            require_once FLOYI_CONNECT_PLUGIN_DIR . 'admin/class-floyi-admin.php';
        }
    }

    /**
     * Initialize hooks.
     */
    private function init_hooks() {
        // Activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Initialize components
        add_action('init', array($this, 'init'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Admin hooks
        if (is_admin()) {
            add_action('admin_menu', array('Floyi_Admin', 'add_menu_page'));
            add_action('admin_init', array('Floyi_Admin', 'register_settings'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
            add_filter('plugin_action_links_' . FLOYI_CONNECT_PLUGIN_BASENAME, array($this, 'add_plugin_action_links'));
        }

        // Webhook retry cron
        add_action('floyi_process_webhook_queue', array('Floyi_Webhook_Queue', 'process_queue'));

        // Track post status changes for Floyi-managed posts
        add_action('transition_post_status', array($this, 'handle_post_status_change'), 10, 3);
    }

    /**
     * Handle post status transitions for Floyi-managed posts.
     *
     * Queues a webhook notification when a post with a _floyi_article_id
     * changes status (e.g., draft → publish, publish → trash).
     *
     * @param string  $new_status New post status.
     * @param string  $old_status Old post status.
     * @param WP_Post $post       Post object.
     */
    public function handle_post_status_change($new_status, $old_status, $post) {
        error_log("[Floyi] transition_post_status fired: post_id={$post->ID}, old={$old_status}, new={$new_status}");

        // Skip if status hasn't actually changed
        if ($new_status === $old_status) {
            error_log("[Floyi] Skipped: status unchanged");
            return;
        }

        // Skip if not connected to Floyi
        if (!self::is_connected()) {
            error_log("[Floyi] Skipped: not connected");
            return;
        }

        // Only track posts that were published via Floyi (have _floyi_article_id meta)
        $floyi_article_id = get_post_meta($post->ID, '_floyi_article_id', true);
        if (!$floyi_article_id) {
            error_log("[Floyi] Skipped: no _floyi_article_id meta on post {$post->ID}");
            return;
        }

        error_log("[Floyi] Queuing webhook: post_id={$post->ID}, floyi_article_id={$floyi_article_id}");

        // Queue webhook event
        $queue_id = Floyi_Webhook_Queue::add('post_status_changed', array(
            'wp_post_id'       => $post->ID,
            'floyi_article_id' => $floyi_article_id,
            'old_status'       => $old_status,
            'new_status'       => $new_status,
            'permalink'        => get_permalink($post->ID),
            'edit_url'         => get_edit_post_link($post->ID, 'raw'),
            'post_title'       => $post->post_title,
        ));

        error_log("[Floyi] Queue result: " . ($queue_id ? "id={$queue_id}" : "FAILED"));

        // Process queue immediately instead of waiting for cron
        Floyi_Webhook_Queue::process_queue();
        error_log("[Floyi] Immediate queue processing triggered");
    }

    /**
     * Plugin initialization.
     */
    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('floyi-connect', false, dirname(FLOYI_CONNECT_PLUGIN_BASENAME) . '/languages');
    }

    /**
     * Register REST API routes.
     */
    public function register_rest_routes() {
        Floyi_API::register_routes();
    }

    /**
     * Add Settings link on the Plugins page.
     *
     * @param array $links Existing plugin action links.
     * @return array Modified action links.
     */
    public function add_plugin_action_links($links) {
        $settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=floyi-connect')) . '">'
            . esc_html__('Settings', 'floyi-connect') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Enqueue admin assets.
     *
     * @param string $hook Current admin page.
     */
    public function enqueue_admin_assets($hook) {
        if ('settings_page_floyi-connect' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'floyi-connect-admin',
            FLOYI_CONNECT_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            FLOYI_CONNECT_VERSION
        );

        wp_enqueue_script(
            'floyi-connect-admin',
            FLOYI_CONNECT_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            FLOYI_CONNECT_VERSION,
            true
        );

        wp_localize_script('floyi-connect-admin', 'floyiConnect', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('floyi_connect_nonce'),
            'strings' => array(
                'connecting' => __('Connecting...', 'floyi-connect'),
                'connected' => __('Connected', 'floyi-connect'),
                'disconnected' => __('Disconnected', 'floyi-connect'),
                'error' => __('Connection failed', 'floyi-connect'),
            ),
        ));
    }

    /**
     * Plugin activation.
     */
    public function activate() {
        // Create database tables for nonce cache and webhook queue
        $this->create_tables();

        // Generate site ID if not exists
        if (!get_option('floyi_site_id')) {
            update_option('floyi_site_id', wp_generate_uuid4());
        }

        // Schedule webhook queue processing
        if (!wp_next_scheduled('floyi_process_webhook_queue')) {
            wp_schedule_event(time(), 'every_minute', 'floyi_process_webhook_queue');
        }

        // Set default options
        $defaults = array(
            'floyi_connection_status' => 'disconnected',
            'floyi_connected_at' => '',
            'floyi_last_sync' => '',
        );

        foreach ($defaults as $option => $value) {
            if (get_option($option) === false) {
                add_option($option, $value);
            }
        }

        // Flush rewrite rules for REST API
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation.
     */
    public function deactivate() {
        // Clear scheduled hooks
        wp_clear_scheduled_hook('floyi_process_webhook_queue');

        // Note: We don't delete data on deactivation (only on uninstall)
    }

    /**
     * Create plugin database tables.
     */
    private function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Nonce cache table
        $nonce_table = $wpdb->prefix . 'floyi_nonce_cache';
        $sql_nonce = "CREATE TABLE IF NOT EXISTS $nonce_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            nonce varchar(64) NOT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY nonce (nonce),
            KEY created_at (created_at)
        ) $charset_collate;";

        // Webhook queue table
        $webhook_table = $wpdb->prefix . 'floyi_webhook_queue';
        $sql_webhook = "CREATE TABLE IF NOT EXISTS $webhook_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            payload longtext NOT NULL,
            attempts int(11) NOT NULL DEFAULT 0,
            max_attempts int(11) NOT NULL DEFAULT 3,
            next_retry datetime DEFAULT NULL,
            last_error text,
            status varchar(20) NOT NULL DEFAULT 'pending',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY next_retry (next_retry)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql_nonce);
        dbDelta($sql_webhook);
    }

    /**
     * Get connection status.
     *
     * @return string Connection status.
     */
    public static function get_connection_status() {
        return get_option('floyi_connection_status', 'disconnected');
    }

    /**
     * Check if connected to Floyi.
     *
     * @return bool True if connected.
     */
    public static function is_connected() {
        return self::get_connection_status() === 'connected';
    }
}

// Add custom cron schedule
add_filter('cron_schedules', function($schedules) {
    $schedules['every_minute'] = array(
        'interval' => 60,
        'display' => __('Every Minute', 'floyi-connect'),
    );
    return $schedules;
});

// Initialize plugin
function floyi_connect() {
    return Floyi_Connect::get_instance();
}

// Start the plugin
floyi_connect();
