/**
 * Floyi Connect Admin JavaScript
 *
 * Handles connection management, code generation, and status updates.
 *
 * @package Floyi_Connect
 */

(function($) {
    'use strict';

    /**
     * Floyi Connect Admin Controller
     */
    var FloyiConnect = {
        /**
         * Initialize the controller
         */
        init: function() {
            this.bindEvents();
            this.checkConnectionStatus();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            $(document).on('click', '.floyi-generate-code', this.generateCode.bind(this));
            $(document).on('click', '.floyi-copy-code', this.copyCode.bind(this));
            $(document).on('click', '.floyi-disconnect', this.disconnect.bind(this));
            $(document).on('click', '.floyi-refresh-status', this.refreshStatus.bind(this));
        },

        /**
         * Generate a new connection code
         */
        generateCode: function(e) {
            e.preventDefault();

            var $button = $(e.currentTarget);
            var originalText = $button.html();

            $button.prop('disabled', true).html(
                '<span class="floyi-spinner"></span> ' + floyiConnect.strings.connecting
            );

            $.ajax({
                url: floyiConnect.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'floyi_generate_code',
                    nonce: floyiConnect.nonce
                },
                success: function(response) {
                    if (response.success) {
                        FloyiConnect.updateCodeDisplay(response.data);
                        FloyiConnect.showNotice('success', 'Connection code generated successfully.');
                    } else {
                        FloyiConnect.showNotice('error', response.data.message || 'Failed to generate code.');
                    }
                },
                error: function() {
                    FloyiConnect.showNotice('error', 'Network error. Please try again.');
                },
                complete: function() {
                    $button.prop('disabled', false).html(originalText);
                }
            });
        },

        /**
         * Copy connection code to clipboard
         */
        copyCode: function(e) {
            e.preventDefault();

            var $codeValue = $('.floyi-code-value');
            var code = $codeValue.text().trim();

            if (!code || code === '----') {
                this.showNotice('warning', 'No code to copy. Generate a code first.');
                return;
            }

            // Try to use the Clipboard API
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(code).then(function() {
                    FloyiConnect.showNotice('success', 'Code copied to clipboard!');
                    FloyiConnect.flashCopied($codeValue);
                }).catch(function() {
                    FloyiConnect.fallbackCopy(code);
                });
            } else {
                this.fallbackCopy(code);
            }
        },

        /**
         * Fallback copy method for older browsers
         */
        fallbackCopy: function(text) {
            var $temp = $('<textarea>');
            $('body').append($temp);
            $temp.val(text).select();

            try {
                document.execCommand('copy');
                this.showNotice('success', 'Code copied to clipboard!');
                this.flashCopied($('.floyi-code-value'));
            } catch (err) {
                this.showNotice('error', 'Failed to copy. Please select and copy manually.');
            }

            $temp.remove();
        },

        /**
         * Flash effect when code is copied
         */
        flashCopied: function($element) {
            $element.css('background', '#d4edda');
            setTimeout(function() {
                $element.css('background', '');
            }, 500);
        },

        /**
         * Disconnect from Floyi
         */
        disconnect: function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to disconnect from Floyi? You will need to reconnect to publish content.')) {
                return;
            }

            var $button = $(e.currentTarget);
            var originalText = $button.html();

            $button.prop('disabled', true).html(
                '<span class="floyi-spinner"></span> Disconnecting...'
            );

            $.ajax({
                url: floyiConnect.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'floyi_disconnect',
                    nonce: floyiConnect.nonce
                },
                success: function(response) {
                    if (response.success) {
                        FloyiConnect.showNotice('success', 'Disconnected successfully.');
                        // Reload to show updated status
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        FloyiConnect.showNotice('error', response.data.message || 'Failed to disconnect.');
                        $button.prop('disabled', false).html(originalText);
                    }
                },
                error: function() {
                    FloyiConnect.showNotice('error', 'Network error. Please try again.');
                    $button.prop('disabled', false).html(originalText);
                }
            });
        },

        /**
         * Refresh connection status
         */
        refreshStatus: function(e) {
            e.preventDefault();

            var $button = $(e.currentTarget);
            var originalText = $button.html();

            $button.prop('disabled', true).html(
                '<span class="floyi-spinner"></span> Checking...'
            );

            $.ajax({
                url: floyiConnect.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'floyi_check_status',
                    nonce: floyiConnect.nonce
                },
                success: function(response) {
                    if (response.success) {
                        FloyiConnect.updateStatusDisplay(response.data);
                    } else {
                        FloyiConnect.showNotice('error', 'Failed to check status.');
                    }
                },
                error: function() {
                    FloyiConnect.showNotice('error', 'Network error. Please try again.');
                },
                complete: function() {
                    $button.prop('disabled', false).html(originalText);
                }
            });
        },

        /**
         * Check connection status on page load
         */
        checkConnectionStatus: function() {
            var $statusIndicator = $('.floyi-status-indicator');

            // If we're in pending state, poll for updates
            if ($statusIndicator.hasClass('pending')) {
                this.startStatusPolling();
            }
        },

        /**
         * Start polling for status updates when code is pending
         */
        startStatusPolling: function() {
            var pollCount = 0;
            var maxPolls = 60; // Poll for 5 minutes (every 5 seconds)

            var pollInterval = setInterval(function() {
                pollCount++;

                if (pollCount >= maxPolls) {
                    clearInterval(pollInterval);
                    return;
                }

                $.ajax({
                    url: floyiConnect.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'floyi_check_status',
                        nonce: floyiConnect.nonce
                    },
                    success: function(response) {
                        if (response.success && response.data.status === 'connected') {
                            clearInterval(pollInterval);
                            FloyiConnect.showNotice('success', 'Connected to Floyi successfully!');
                            setTimeout(function() {
                                location.reload();
                            }, 1500);
                        }
                    }
                });
            }, 5000);
        },

        /**
         * Update the code display area
         */
        updateCodeDisplay: function(data) {
            var $codeValue = $('.floyi-code-value');
            var $codeExpiry = $('.floyi-code-expiry');

            if (data.code) {
                $codeValue.text(data.code);
            }

            if (data.expires_in) {
                $codeExpiry.removeClass('expired').text('Expires in ' + data.expires_in);
            }

            // Update status to pending
            var $statusIndicator = $('.floyi-status-indicator');
            $statusIndicator
                .removeClass('connected disconnected')
                .addClass('pending');
            $statusIndicator.find('.floyi-status-text').text('Pending Connection');

            // Start polling
            this.startStatusPolling();
        },

        /**
         * Update the status display
         */
        updateStatusDisplay: function(data) {
            var $statusIndicator = $('.floyi-status-indicator');
            var $statusText = $statusIndicator.find('.floyi-status-text');

            $statusIndicator.removeClass('connected disconnected pending');
            $statusIndicator.addClass(data.status);

            var statusLabels = {
                'connected': floyiConnect.strings.connected,
                'disconnected': floyiConnect.strings.disconnected,
                'pending': 'Pending Connection'
            };

            $statusText.text(statusLabels[data.status] || data.status);

            if (data.connected_at) {
                $('.floyi-connected-at').text(data.connected_at);
            }

            if (data.last_sync) {
                $('.floyi-last-sync').text(data.last_sync);
            }
        },

        /**
         * Show a notice message
         */
        showNotice: function(type, message) {
            var iconMap = {
                'success': 'dashicons-yes-alt',
                'error': 'dashicons-warning',
                'warning': 'dashicons-info',
                'info': 'dashicons-info-outline'
            };

            var $notice = $('<div class="floyi-notice floyi-notice-' + type + '">' +
                '<span class="dashicons ' + iconMap[type] + '"></span>' +
                '<span>' + message + '</span>' +
                '</div>');

            // Remove any existing notices
            $('.floyi-notice').remove();

            // Insert notice at the top of the settings page
            $('.floyi-connect-settings').prepend($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        FloyiConnect.init();
    });

})(jQuery);
