<?php
/**
 * Admin class for Floyi Connect.
 *
 * Handles the WordPress admin settings page.
 *
 * @package Floyi_Connect
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin class.
 */
class Floyi_Admin {

    /**
     * Add admin menu page.
     */
    public static function add_menu_page() {
        add_options_page(
            __('Floyi Connect', 'floyi-connect'),
            __('Floyi', 'floyi-connect'),
            'manage_options',
            'floyi-connect',
            array(__CLASS__, 'render_settings_page')
        );
    }

    /**
     * Register settings.
     */
    public static function register_settings() {
        // Handle AJAX actions
        add_action('wp_ajax_floyi_connect', array(__CLASS__, 'ajax_connect'));
        add_action('wp_ajax_floyi_disconnect', array(__CLASS__, 'ajax_disconnect'));
        add_action('wp_ajax_floyi_retry_webhook', array(__CLASS__, 'ajax_retry_webhook'));
    }

    /**
     * Render settings page.
     */
    public static function render_settings_page() {
        $settings = Floyi_Settings::get_all_settings();
        $webhook_status = Floyi_Webhook_Queue::get_status();
        ?>
        <div class="wrap floyi-settings">
            <h1><?php echo esc_html__('Floyi Connect', 'floyi-connect'); ?></h1>

            <div class="floyi-card">
                <h2><?php echo esc_html__('Connection Status', 'floyi-connect'); ?></h2>

                <?php if ($settings['is_connected']) : ?>
                    <div class="floyi-status floyi-status-connected">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php echo esc_html__('Connected to Floyi', 'floyi-connect'); ?>
                    </div>

                    <table class="form-table">
                        <tr>
                            <th><?php echo esc_html__('Site ID', 'floyi-connect'); ?></th>
                            <td><code><?php echo esc_html($settings['site_id']); ?></code></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Connected Since', 'floyi-connect'); ?></th>
                            <td><?php echo esc_html($settings['connected_at']); ?></td>
                        </tr>
                        <tr>
                            <th><?php echo esc_html__('Last Sync', 'floyi-connect'); ?></th>
                            <td><?php echo $settings['last_sync'] ? esc_html($settings['last_sync']) : esc_html__('Never', 'floyi-connect'); ?></td>
                        </tr>
                    </table>

                    <p>
                        <button type="button" class="button button-secondary" id="floyi-disconnect">
                            <?php echo esc_html__('Disconnect', 'floyi-connect'); ?>
                        </button>
                    </p>

                <?php else : ?>
                    <div class="floyi-status floyi-status-disconnected">
                        <span class="dashicons dashicons-warning"></span>
                        <?php echo esc_html__('Not connected', 'floyi-connect'); ?>
                    </div>

                    <h3><?php echo esc_html__('Connect to Floyi', 'floyi-connect'); ?></h3>

                    <ol class="floyi-steps">
                        <li><?php echo esc_html__('In Floyi, go to Settings > Integrations', 'floyi-connect'); ?></li>
                        <li><?php echo esc_html__('Click "Connect Site" under WordPress', 'floyi-connect'); ?></li>
                        <li><?php echo esc_html__('Copy the connection token from Floyi', 'floyi-connect'); ?></li>
                        <li><?php echo esc_html__('Paste the token below and click "Connect"', 'floyi-connect'); ?></li>
                    </ol>

                    <div class="floyi-connection-form">
                        <label for="floyi-token" class="floyi-label">
                            <?php echo esc_html__('Connection Token from Floyi', 'floyi-connect'); ?>
                        </label>
                        <div class="floyi-input-group">
                            <input
                                type="text"
                                id="floyi-token"
                                class="floyi-token-input"
                                placeholder="XXXX-XXXX-XXXX"
                                maxlength="20"
                                autocomplete="off"
                            />
                            <button type="button" class="button button-primary" id="floyi-connect-btn">
                                <?php echo esc_html__('Connect', 'floyi-connect'); ?>
                            </button>
                        </div>
                        <p class="floyi-hint">
                            <?php echo esc_html__('The token looks like: ABCD-EFGH-IJKL', 'floyi-connect'); ?>
                        </p>
                        <div id="floyi-connect-error" class="floyi-error" style="display: none;"></div>
                    </div>
                <?php endif; ?>
            </div>

            <?php if ($settings['is_connected']) : ?>
            <div class="floyi-card">
                <h2><?php echo esc_html__('Site Capabilities', 'floyi-connect'); ?></h2>
                <p><?php echo esc_html__('These capabilities are shared with Floyi for content publishing.', 'floyi-connect'); ?></p>

                <?php
                $capabilities = Floyi_Capabilities::get_all();
                ?>

                <table class="form-table">
                    <tr>
                        <th><?php echo esc_html__('Post Types', 'floyi-connect'); ?></th>
                        <td>
                            <?php
                            $type_names = array_map(function($t) { return $t['name']; }, $capabilities['post_types']);
                            echo esc_html(implode(', ', $type_names));
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Categories', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(count($capabilities['categories'])); ?> <?php echo esc_html__('available', 'floyi-connect'); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Tags', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(count($capabilities['tags'])); ?> <?php echo esc_html__('available', 'floyi-connect'); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Authors', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(count($capabilities['authors'])); ?> <?php echo esc_html__('available', 'floyi-connect'); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('SEO Plugin', 'floyi-connect'); ?></th>
                        <td>
                            <?php
                            if ($capabilities['seo_plugin']) {
                                echo esc_html($capabilities['seo_plugin']['name'] . ' ' . $capabilities['seo_plugin']['version']);
                            } else {
                                echo esc_html__('None detected', 'floyi-connect');
                            }
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Editor', 'floyi-connect'); ?></th>
                        <td>
                            <?php
                            echo $capabilities['editor_type'] === 'block'
                                ? esc_html__('Block Editor (Gutenberg)', 'floyi-connect')
                                : esc_html__('Classic Editor', 'floyi-connect');
                            ?>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="floyi-card">
                <h2><?php echo esc_html__('Webhook Queue', 'floyi-connect'); ?></h2>
                <p><?php echo esc_html__('Status of webhook notifications to Floyi.', 'floyi-connect'); ?></p>

                <table class="form-table">
                    <tr>
                        <th><?php echo esc_html__('Pending', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html($webhook_status['pending']); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Retrying', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html($webhook_status['retrying']); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Failed', 'floyi-connect'); ?></th>
                        <td>
                            <?php echo esc_html($webhook_status['failed']); ?>
                            <?php if ($webhook_status['failed'] > 0) : ?>
                                <button type="button" class="button button-small" id="floyi-show-failed">
                                    <?php echo esc_html__('View', 'floyi-connect'); ?>
                                </button>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
            </div>
            <?php endif; ?>

            <div class="floyi-card">
                <h2><?php echo esc_html__('Site Information', 'floyi-connect'); ?></h2>
                <table class="form-table">
                    <tr>
                        <th><?php echo esc_html__('WordPress Version', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(get_bloginfo('version')); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('PHP Version', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(phpversion()); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Plugin Version', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(FLOYI_CONNECT_VERSION); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Site URL', 'floyi-connect'); ?></th>
                        <td><?php echo esc_html(get_site_url()); ?></td>
                    </tr>
                </table>
            </div>
        </div>

        <style>
            .floyi-settings { max-width: 800px; }
            .floyi-card {
                background: #fff;
                border: 1px solid #c3c4c7;
                border-radius: 4px;
                padding: 20px;
                margin-bottom: 20px;
            }
            .floyi-card h2 { margin-top: 0; }
            .floyi-status {
                padding: 10px 15px;
                border-radius: 4px;
                display: inline-flex;
                align-items: center;
                gap: 8px;
                margin-bottom: 15px;
            }
            .floyi-status-connected { background: #d4edda; color: #155724; }
            .floyi-status-disconnected { background: #fff3cd; color: #856404; }
            .floyi-steps { margin-left: 20px; margin-bottom: 20px; }
            .floyi-steps li { margin-bottom: 8px; }
            .floyi-connection-form { margin-top: 20px; }
            .floyi-label {
                display: block;
                font-weight: 600;
                margin-bottom: 8px;
            }
            .floyi-input-group {
                display: flex;
                gap: 10px;
                align-items: center;
            }
            .floyi-token-input {
                font-size: 18px;
                font-family: monospace;
                letter-spacing: 2px;
                padding: 10px 15px;
                width: 220px;
                text-transform: uppercase;
                border: 2px solid #c3c4c7;
                border-radius: 4px;
            }
            .floyi-token-input:focus {
                border-color: #2271b1;
                outline: none;
                box-shadow: 0 0 0 1px #2271b1;
            }
            .floyi-hint {
                color: #666;
                font-size: 13px;
                margin-top: 8px;
            }
            .floyi-error {
                background: #f8d7da;
                color: #721c24;
                padding: 10px 15px;
                border-radius: 4px;
                margin-top: 15px;
            }
            #floyi-connect-btn {
                padding: 8px 20px;
                height: auto;
            }
        </style>

        <script>
            jQuery(document).ready(function($) {
                // Connect to Floyi
                $('#floyi-connect-btn').on('click', function() {
                    var $btn = $(this);
                    var $input = $('#floyi-token');
                    var $error = $('#floyi-connect-error');
                    var token = $input.val().trim().toUpperCase();

                    // Validate token format
                    if (!token) {
                        $error.text('<?php echo esc_js(__('Please enter the connection token from Floyi.', 'floyi-connect')); ?>').show();
                        return;
                    }

                    if (!/^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/.test(token)) {
                        $error.text('<?php echo esc_js(__('Invalid token format. The token should look like: ABCD-EFGH-IJKL', 'floyi-connect')); ?>').show();
                        return;
                    }

                    $error.hide();
                    $btn.prop('disabled', true).text('<?php echo esc_js(__('Connecting...', 'floyi-connect')); ?>');
                    $input.prop('disabled', true);

                    $.post(ajaxurl, {
                        action: 'floyi_connect',
                        nonce: '<?php echo wp_create_nonce('floyi_admin_nonce'); ?>',
                        token: token
                    }, function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            $error.text(response.data.message || '<?php echo esc_js(__('Failed to connect. Please check the token and try again.', 'floyi-connect')); ?>').show();
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Connect', 'floyi-connect')); ?>');
                            $input.prop('disabled', false);
                        }
                    }).fail(function() {
                        $error.text('<?php echo esc_js(__('Connection failed. Please check your internet connection and try again.', 'floyi-connect')); ?>').show();
                        $btn.prop('disabled', false).text('<?php echo esc_js(__('Connect', 'floyi-connect')); ?>');
                        $input.prop('disabled', false);
                    });
                });

                // Allow Enter key to submit
                $('#floyi-token').on('keypress', function(e) {
                    if (e.which === 13) {
                        $('#floyi-connect-btn').click();
                    }
                });

                // Auto-format token input (add dashes)
                $('#floyi-token').on('input', function() {
                    var val = $(this).val().toUpperCase().replace(/[^A-Z0-9]/g, '');
                    if (val.length > 4) {
                        val = val.substring(0, 4) + '-' + val.substring(4);
                    }
                    if (val.length > 9) {
                        val = val.substring(0, 9) + '-' + val.substring(9);
                    }
                    if (val.length > 14) {
                        val = val.substring(0, 14);
                    }
                    $(this).val(val);
                });

                // Disconnect
                $('#floyi-disconnect').on('click', function() {
                    if (!confirm('<?php echo esc_js(__('Are you sure you want to disconnect from Floyi?', 'floyi-connect')); ?>')) {
                        return;
                    }

                    var $btn = $(this);
                    $btn.prop('disabled', true).text('<?php echo esc_js(__('Disconnecting...', 'floyi-connect')); ?>');

                    $.post(ajaxurl, {
                        action: 'floyi_disconnect',
                        nonce: '<?php echo wp_create_nonce('floyi_admin_nonce'); ?>'
                    }, function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Failed to disconnect', 'floyi-connect')); ?>');
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Disconnect', 'floyi-connect')); ?>');
                        }
                    });
                });
            });
        </script>
        <?php
    }

    /**
     * AJAX: Connect to Floyi using token.
     */
    public static function ajax_connect() {
        check_ajax_referer('floyi_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'floyi-connect')));
        }

        $token = isset($_POST['token']) ? sanitize_text_field($_POST['token']) : '';

        if (empty($token)) {
            wp_send_json_error(array('message' => __('Token is required.', 'floyi-connect')));
        }

        // Validate token format
        if (!preg_match('/^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $token)) {
            wp_send_json_error(array('message' => __('Invalid token format.', 'floyi-connect')));
        }

        // Get site info and capabilities
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        $site_id = wp_generate_uuid4();
        $current_user = wp_get_current_user();
        $capabilities = Floyi_Capabilities::get_all();

        // Generate site token and webhook secret for secure communication
        $site_token = wp_generate_password(64, false);
        $webhook_secret = wp_generate_password(32, false);

        // Resolve API URL: FLOYI_API_URL constant (wp-config.php) > production default
        $floyi_api_url = Floyi_Settings::get_floyi_url();

        // Call Floyi API to verify the handshake
        $verify_url = $floyi_api_url . '/api/cms/wordpress/handshake/verify/';

        $response = wp_remote_post($verify_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'token' => $token,
                'site_url' => $site_url,
                'site_name' => $site_name,
                'wp_site_id' => $site_id,
                'wp_user_email' => $current_user->user_email,
                'site_token' => $site_token,
                'webhook_secret' => $webhook_secret,
                'capabilities' => $capabilities,
                'seo_plugin' => $capabilities['seo_plugin'],
            )),
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => sprintf(
                    __('Connection failed: %s', 'floyi-connect'),
                    $response->get_error_message()
                )
            ));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code !== 201) {
            $error_message = isset($data['detail']) ? $data['detail'] : __('Connection failed. Please check the token and try again.', 'floyi-connect');
            wp_send_json_error(array('message' => $error_message));
        }

        // Save connection settings
        Floyi_Settings::mark_connected(
            $site_id,
            $site_token,
            $webhook_secret,
            $floyi_api_url  // Store the resolved URL for reference
        );

        wp_send_json_success(array(
            'message' => __('Successfully connected to Floyi!', 'floyi-connect'),
            'connection_id' => isset($data['connection_id']) ? $data['connection_id'] : null,
        ));
    }

    /**
     * AJAX: Disconnect from Floyi.
     */
    public static function ajax_disconnect() {
        check_ajax_referer('floyi_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'floyi-connect')));
        }

        // Notify Floyi backend before clearing local credentials (best-effort).
        self::notify_floyi_disconnect();

        Floyi_Settings::mark_disconnected();

        wp_send_json_success();
    }

    /**
     * Notify the Floyi backend that this site is disconnecting.
     *
     * Sends a webhook with event 'site_disconnected' so Floyi marks its
     * side of the connection as disconnected. Called before tokens are
     * cleared so authentication still works.
     */
    private static function notify_floyi_disconnect() {
        $floyi_url = Floyi_Settings::get_floyi_url();
        $site_token = Floyi_Settings::get_site_token();
        $webhook_secret = Floyi_Settings::get_webhook_secret();

        if (!$site_token || !$webhook_secret) {
            return; // Not connected — nothing to notify.
        }

        $webhook_url = trailingslashit($floyi_url) . 'api/cms/wordpress/webhook/receive/';

        $body = wp_json_encode(array(
            'event' => 'site_disconnected',
            'data' => array(),
            'site_id' => Floyi_Settings::get_site_id(),
            'timestamp' => current_time('c'),
        ));

        $timestamp = (string) time();
        $nonce = bin2hex(random_bytes(16));
        $signature = Floyi_Security::generate_signature($body, $timestamp, $nonce, $webhook_secret);

        wp_remote_post($webhook_url, array(
            'timeout' => 10,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Floyi-Token' => $site_token,
                'X-Floyi-Timestamp' => $timestamp,
                'X-Floyi-Nonce' => $nonce,
                'X-Floyi-Signature' => $signature,
            ),
            'body' => $body,
            'sslverify' => true,
        ));
        // Intentionally ignoring the response — disconnect should proceed regardless.
    }

    /**
     * AJAX: Retry failed webhook.
     */
    public static function ajax_retry_webhook() {
        check_ajax_referer('floyi_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'floyi-connect')));
        }

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

        if (!$id) {
            wp_send_json_error(array('message' => __('Invalid webhook ID.', 'floyi-connect')));
        }

        $result = Floyi_Webhook_Queue::retry_item($id);

        if ($result) {
            wp_send_json_success();
        } else {
            wp_send_json_error(array('message' => __('Failed to retry webhook.', 'floyi-connect')));
        }
    }
}
